#!/usr/bin/env php
<?php
/**
 * Image Storage Migration Verification Script
 * Run: php verify-migration.php
 */

echo "\n╔════════════════════════════════════════════════════════════════╗\n";
echo "║         IMAGE STORAGE MIGRATION VERIFICATION                  ║\n";
echo "╚════════════════════════════════════════════════════════════════╝\n\n";

$baseDir = __DIR__;
$checks = [];
$errors = [];
$warnings = [];

// Check 1: public/uploads directory exists
echo "Checking directory structure...\n";
$uploadsDir = $baseDir . '/public/uploads';
if (is_dir($uploadsDir)) {
    $checks[] = "✓ public/uploads directory exists";
    
    // Check subdirectories
    $subdirs = [
        'profile-images', 'cars', 'settings', 
        'payment-methods/logos', 'payment-methods/qrcodes',
        'kyc/documents', 'kyc/selfies', 'investment-plans'
    ];
    
    foreach ($subdirs as $subdir) {
        if (is_dir($uploadsDir . '/' . $subdir)) {
            $checks[] = "  ✓ {$subdir}/ exists";
        } else {
            $warnings[] = "  ⚠ {$subdir}/ not found (may be created on first upload)";
        }
    }
} else {
    $errors[] = "✗ public/uploads directory not found!";
}

// Check 2: .htaccess file
echo "\nChecking .htaccess configuration...\n";
$htaccess = $uploadsDir . '/.htaccess';
if (file_exists($htaccess)) {
    $content = file_get_contents($htaccess);
    if (strpos($content, 'Options -Indexes') !== false) {
        $checks[] = "✓ .htaccess exists with proper configuration";
    } else {
        $warnings[] = "⚠ .htaccess exists but may be misconfigured";
    }
} else {
    $warnings[] = "⚠ .htaccess not found in uploads directory";
}

// Check 3: Filesystem configuration
echo "\nChecking filesystem configuration...\n";
$filesystemsConfig = $baseDir . '/config/filesystems.php';
if (file_exists($filesystemsConfig)) {
    $content = file_get_contents($filesystemsConfig);
    if (strpos($content, "'public_uploads'") !== false) {
        $checks[] = "✓ public_uploads disk configured";
    } else {
        $errors[] = "✗ public_uploads disk NOT found in config/filesystems.php!";
    }
} else {
    $errors[] = "✗ config/filesystems.php not found!";
}

// Check 4: Controllers updated
echo "\nChecking controller updates...\n";
$controllers = [
    'app/Http/Controllers/ProfileController.php',
    'app/Http/Controllers/Admin/CarController.php',
    'app/Http/Controllers/Admin/SettingController.php',
    'app/Http/Controllers/Admin/PaymentMethodController.php',
    'app/Http/Controllers/KYCController.php',
    'app/Http/Controllers/Admin/InvestmentPlanController.php',
];

foreach ($controllers as $controller) {
    $path = $baseDir . '/' . $controller;
    if (file_exists($path)) {
        $content = file_get_contents($path);
        if (strpos($content, 'public_uploads') !== false) {
            $checks[] = "✓ " . basename($controller) . " updated";
        } else {
            $errors[] = "✗ " . basename($controller) . " NOT updated!";
        }
    }
}

// Check 5: Helper updated
echo "\nChecking helper functions...\n";
$helperPath = $baseDir . '/app/Helpers/AppHelper.php';
if (file_exists($helperPath)) {
    $content = file_get_contents($helperPath);
    if (strpos($content, "asset('uploads/'") !== false) {
        $checks[] = "✓ AppHelper updated (siteLogo, siteFavicon)";
    } else {
        $errors[] = "✗ AppHelper NOT updated!";
    }
}

// Check 6: Sample blade files
echo "\nChecking blade templates (sample)...\n";
$sampleBlades = [
    'resources/views/profile/edit.blade.php',
    'resources/views/wallet/deposit.blade.php',
    'resources/views/frontend/home.blade.php',
];

foreach ($sampleBlades as $blade) {
    $path = $baseDir . '/' . $blade;
    if (file_exists($path)) {
        $content = file_get_contents($path);
        $storageCount = substr_count($content, "asset('storage/'");
        $uploadsCount = substr_count($content, "asset('uploads/'");
        
        if ($storageCount > 0) {
            $errors[] = "✗ " . basename($blade) . " still has asset('storage/') references!";
        } elseif ($uploadsCount > 0) {
            $checks[] = "✓ " . basename($blade) . " updated";
        }
    }
}

// Check 7: Migration commands exist
echo "\nChecking migration commands...\n";
$commands = [
    'app/Console/Commands/MigrateImagesToPublic.php',
    'app/Console/Commands/CleanupOldStorage.php',
];

foreach ($commands as $command) {
    $path = $baseDir . '/' . $command;
    if (file_exists($path)) {
        $checks[] = "✓ " . basename($command) . " available";
    } else {
        $warnings[] = "⚠ " . basename($command) . " not found";
    }
}

// Check 8: Count files
echo "\nCounting migrated files...\n";
$fileCount = 0;
if (is_dir($uploadsDir)) {
    $iterator = new RecursiveIteratorIterator(new RecursiveDirectoryIterator($uploadsDir));
    foreach ($iterator as $file) {
        if ($file->isFile() && $file->getFilename() !== '.htaccess' && $file->getFilename() !== '.gitkeep') {
            $fileCount++;
        }
    }
    $checks[] = "✓ Found {$fileCount} uploaded files";
}

// Check 9: Permissions
echo "\nChecking permissions...\n";
if (is_writable($uploadsDir)) {
    $checks[] = "✓ uploads directory is writable";
} else {
    $errors[] = "✗ uploads directory is NOT writable!";
}

// Print Results
echo "\n" . str_repeat("=", 66) . "\n";
echo "VERIFICATION RESULTS\n";
echo str_repeat("=", 66) . "\n\n";

if (!empty($checks)) {
    echo "✅ PASSED CHECKS (" . count($checks) . "):\n";
    foreach ($checks as $check) {
        echo "  {$check}\n";
    }
    echo "\n";
}

if (!empty($warnings)) {
    echo "⚠️  WARNINGS (" . count($warnings) . "):\n";
    foreach ($warnings as $warning) {
        echo "  {$warning}\n";
    }
    echo "\n";
}

if (!empty($errors)) {
    echo "❌ ERRORS (" . count($errors) . "):\n";
    foreach ($errors as $error) {
        echo "  {$error}\n";
    }
    echo "\n";
    echo "Migration may not be complete. Please review errors above.\n\n";
    exit(1);
} else {
    echo "✅ MIGRATION VERIFIED SUCCESSFULLY!\n\n";
    echo "Next steps:\n";
    echo "  1. Test image uploads in the application\n";
    echo "  2. Verify images display correctly\n";
    echo "  3. Run: php artisan images:cleanup-old-storage\n\n";
    exit(0);
}
