<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;

class MembershipTier extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'slug',
        'description',
        'price',
        'currency',
        'duration_days',
        'benefits',
        'car_discount_percentage',
        'investment_bonus_percentage',
        'giveaway_entry_multiplier',
        'priority_support',
        'is_active',
        'sort_order',
    ];

    protected $casts = [
        'price' => 'decimal:2',
        'car_discount_percentage' => 'decimal:2',
        'investment_bonus_percentage' => 'decimal:2',
        'giveaway_entry_multiplier' => 'integer',
        'priority_support' => 'boolean',
        'is_active' => 'boolean',
        'benefits' => 'array',
        'duration_days' => 'integer',
        'sort_order' => 'integer',
    ];

    /**
     * Boot method to auto-generate slug
     */
    protected static function boot()
    {
        parent::boot();
        
        static::creating(function ($tier) {
            if (empty($tier->slug)) {
                $tier->slug = Str::slug($tier->name);
            }
        });
    }

    /**
     * Relationships
     */
    public function cards()
    {
        return $this->hasMany(MembershipCard::class);
    }

    public function purchases()
    {
        return $this->hasMany(MembershipPurchase::class);
    }

    /**
     * Scopes
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeOrdered($query)
    {
        return $query->orderBy('sort_order', 'asc');
    }

    /**
     * Accessors
     */
    public function getFormattedPriceAttribute()
    {
        return currency_symbol() . number_format($this->price, 2);
    }

    public function getDurationInMonthsAttribute()
    {
        return round($this->duration_days / 30, 1);
    }

    /**
     * Business Logic Methods
     */
    public function calculateDiscountedPrice($originalPrice)
    {
        $discount = ($originalPrice * $this->car_discount_percentage) / 100;
        return $originalPrice - $discount;
    }

    public function calculateInvestmentBonus($investmentAmount)
    {
        return ($investmentAmount * $this->investment_bonus_percentage) / 100;
    }
}
