<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class MembershipPurchase extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'membership_tier_id',
        'membership_card_id',
        'transaction_id',
        'amount',
        'currency',
        'payment_method',
        'status',
        'notes',
        'purchased_at',
    ];

    protected $casts = [
        'amount' => 'decimal:2',
        'purchased_at' => 'datetime',
    ];

    /**
     * Boot method to auto-generate transaction ID
     */
    protected static function boot()
    {
        parent::boot();
        
        static::creating(function ($purchase) {
            if (empty($purchase->transaction_id)) {
                $purchase->transaction_id = self::generateTransactionId();
            }
        });
    }

    /**
     * Relationships
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function membershipTier()
    {
        return $this->belongsTo(MembershipTier::class);
    }

    public function membershipCard()
    {
        return $this->belongsTo(MembershipCard::class);
    }

    /**
     * Scopes
     */
    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    /**
     * Accessors
     */
    public function getFormattedAmountAttribute()
    {
        return currency_symbol() . number_format($this->amount, 2);
    }

    public function getStatusBadgeAttribute()
    {
        return match($this->status) {
            'completed' => 'bg-green-100 text-green-800',
            'pending' => 'bg-orange-100 text-orange-800',
            'failed' => 'bg-red-100 text-red-800',
            'refunded' => 'bg-gray-100 text-gray-800',
            default => 'bg-gray-100 text-gray-800',
        };
    }

    public function getPaymentMethodLabelAttribute()
    {
        return match($this->payment_method) {
            'wallet' => 'Wallet Balance',
            'crypto' => 'Cryptocurrency',
            'bank_transfer' => 'Bank Transfer',
            default => ucfirst($this->payment_method),
        };
    }

    /**
     * Business Logic Methods
     */
    public static function generateTransactionId()
    {
        do {
            $transactionId = 'MEMBERSHIP-' . strtoupper(substr(md5(uniqid(rand(), true)), 0, 10));
        } while (self::where('transaction_id', $transactionId)->exists());
        
        return $transactionId;
    }

    public function markAsCompleted()
    {
        $this->status = 'completed';
        $this->purchased_at = now();
        $this->save();
    }

    public function markAsFailed($notes = null)
    {
        $this->status = 'failed';
        if ($notes) {
            $this->notes = $notes;
        }
        $this->save();
    }
}
