<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class MembershipCard extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'membership_tier_id',
        'card_number',
        'qr_code_path',
        'issued_at',
        'expires_at',
        'status',
    ];

    protected $casts = [
        'issued_at' => 'date',
        'expires_at' => 'date',
    ];

    /**
     * Boot method to auto-generate card number
     */
    protected static function boot()
    {
        parent::boot();
        
        static::creating(function ($card) {
            if (empty($card->card_number)) {
                $card->card_number = self::generateCardNumber();
            }
            
            if (empty($card->issued_at)) {
                $card->issued_at = now();
            }
        });
    }

    /**
     * Relationships
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function membershipTier()
    {
        return $this->belongsTo(MembershipTier::class);
    }

    public function purchase()
    {
        return $this->hasOne(MembershipPurchase::class);
    }

    /**
     * Scopes
     */
    public function scopeActive($query)
    {
        return $query->where('status', 'active')
                    ->where('expires_at', '>', now());
    }

    public function scopeExpired($query)
    {
        return $query->where('expires_at', '<=', now())
                    ->orWhere('status', 'expired');
    }

    /**
     * Accessors
     */
    public function getQrCodeUrlAttribute()
    {
        if ($this->qr_code_path && file_exists(public_path('uploads/' . $this->qr_code_path))) {
            return asset('uploads/' . $this->qr_code_path);
        }
        return null;
    }

    public function getIsActiveAttribute()
    {
        return $this->status === 'active' && $this->expires_at > now();
    }

    public function getIsExpiredAttribute()
    {
        return $this->expires_at <= now();
    }

    public function getDaysUntilExpiryAttribute()
    {
        if ($this->is_expired) {
            return 0;
        }
        return now()->diffInDays($this->expires_at);
    }

    public function getStatusBadgeAttribute()
    {
        return match($this->status) {
            'active' => 'bg-green-100 text-green-800',
            'expired' => 'bg-gray-100 text-gray-800',
            'cancelled' => 'bg-red-100 text-red-800',
            default => 'bg-gray-100 text-gray-800',
        };
    }

    /**
     * Business Logic Methods
     */
    public static function generateCardNumber()
    {
        do {
            $cardNumber = 'TESLA-' . strtoupper(substr(md5(uniqid(rand(), true)), 0, 12));
        } while (self::where('card_number', $cardNumber)->exists());
        
        return $cardNumber;
    }

    public function generateQrCode()
    {
        // This will be called after purchase to generate QR code
        // Using SimpleSoftwareIO/simple-qrcode or endroid/qr-code
        $qrCodePath = 'membership-cards/' . $this->card_number . '.png';
        
        // For now, just store the path - actual QR generation will be in controller
        $this->qr_code_path = $qrCodePath;
        $this->save();
        
        return $qrCodePath;
    }

    public function markAsExpired()
    {
        $this->status = 'expired';
        $this->save();
    }

    public function cancel()
    {
        $this->status = 'cancelled';
        $this->save();
    }
}
