<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class GiveawayEntry extends Model
{
    use HasFactory;

    protected $fillable = [
        'giveaway_id',
        'user_id',
        'entries_count',
        'entry_source',
        'source_purchase_id',
        'entered_at',
    ];

    protected $casts = [
        'entries_count' => 'integer',
        'entered_at' => 'datetime',
    ];

    /**
     * Boot method
     */
    protected static function boot()
    {
        parent::boot();
        
        static::creating(function ($entry) {
            if (empty($entry->entered_at)) {
                $entry->entered_at = now();
            }
            
            // Apply VIP multiplier if user has active membership
            if ($entry->user && $entry->user->hasActiveMembership()) {
                $membership = $entry->user->getActiveMembershipCard();
                if ($membership) {
                    $multiplier = $membership->membershipTier->giveaway_entry_multiplier ?? 1;
                    $entry->entries_count = ($entry->entries_count ?? 1) * $multiplier;
                }
            }
        });
    }

    /**
     * Relationships
     */
    public function giveaway()
    {
        return $this->belongsTo(Giveaway::class);
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function sourcePurchase()
    {
        return $this->belongsTo(Purchase::class, 'source_purchase_id');
    }

    /**
     * Accessors
     */
    public function getEntrySourceLabelAttribute()
    {
        return match($this->entry_source) {
            'manual' => 'Manual Entry',
            'car_purchase' => 'Automatic (Car Purchase)',
            'membership_bonus' => 'VIP Membership Bonus',
            default => ucfirst($this->entry_source),
        };
    }

    public function getMultiplierBadgeAttribute()
    {
        if ($this->entries_count > 1) {
            return $this->entries_count . 'x';
        }
        return '1x';
    }
}
