<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class Giveaway extends Model
{
    use HasFactory;

    protected $fillable = [
        'title',
        'description',
        'image_path',
        'prize_type',
        'prize_value',
        'currency',
        'prize_car_id',
        'prize_details',
        'max_entries',
        'entry_requirement',
        'start_date',
        'end_date',
        'winner_user_id',
        'winner_selected_at',
        'status',
        'is_featured',
    ];

    protected $casts = [
        'prize_value' => 'decimal:2',
        'start_date' => 'datetime',
        'end_date' => 'datetime',
        'winner_selected_at' => 'datetime',
        'is_featured' => 'boolean',
    ];

    /**
     * Relationships
     */
    public function prizeCar()
    {
        return $this->belongsTo(Car::class, 'prize_car_id');
    }

    public function winner()
    {
        return $this->belongsTo(User::class, 'winner_user_id');
    }

    public function entries()
    {
        return $this->hasMany(GiveawayEntry::class);
    }

    /**
     * Scopes
     */
    public function scopeActive($query)
    {
        return $query->where('status', 'active')
                    ->where('start_date', '<=', now())
                    ->where('end_date', '>', now());
    }

    public function scopeUpcoming($query)
    {
        return $query->where('status', 'active')
                    ->where('start_date', '>', now());
    }

    public function scopeEnded($query)
    {
        return $query->where('status', 'ended')
                    ->orWhere(function($q) {
                        $q->where('status', 'active')
                          ->where('end_date', '<=', now());
                    });
    }

    public function scopeFeatured($query)
    {
        return $query->where('is_featured', true);
    }

    /**
     * Accessors
     */
    public function getImageUrlAttribute()
    {
        if ($this->image_path && file_exists(public_path('uploads/' . $this->image_path))) {
            return asset('uploads/' . $this->image_path);
        }
        return asset('images/default-giveaway.jpg');
    }

    public function getFormattedPrizeValueAttribute()
    {
        if ($this->prize_value) {
            return currency_symbol() . number_format($this->prize_value, 2);
        }
        return 'N/A';
    }

    public function getTotalEntriesAttribute()
    {
        return $this->entries()->sum('entries_count');
    }

    public function getUniqueParticipantsAttribute()
    {
        return $this->entries()->distinct('user_id')->count('user_id');
    }

    public function getIsActiveAttribute()
    {
        return $this->status === 'active' 
            && $this->start_date <= now() 
            && $this->end_date > now();
    }

    public function getIsEndedAttribute()
    {
        return $this->status === 'ended' || $this->end_date <= now();
    }

    public function getHasWinnerAttribute()
    {
        return !is_null($this->winner_user_id);
    }

    public function getDaysRemainingAttribute()
    {
        if ($this->is_ended) {
            return 0;
        }
        return max(0, now()->diffInDays($this->end_date, false));
    }

    public function getStatusBadgeAttribute()
    {
        return match($this->status) {
            'active' => 'bg-green-100 text-green-800',
            'draft' => 'bg-gray-100 text-gray-800',
            'ended' => 'bg-red-100 text-red-800',
            'cancelled' => 'bg-red-100 text-red-800',
            default => 'bg-gray-100 text-gray-800',
        };
    }

    public function getEntryRequirementLabelAttribute()
    {
        return match($this->entry_requirement) {
            'none' => 'Anyone Can Enter',
            'membership' => 'VIP Members Only',
            'purchase' => 'Car Purchase Required',
            'both' => 'VIP Members Who Purchased',
            default => ucfirst($this->entry_requirement),
        };
    }

    /**
     * Business Logic Methods
     */
    public function canUserEnter(User $user)
    {
        // Check if giveaway is active
        if (!$this->is_active) {
            return false;
        }

        // Check if user already entered
        if ($this->entries()->where('user_id', $user->id)->exists()) {
            return false;
        }

        // Check max entries limit
        if ($this->max_entries && $this->total_entries >= $this->max_entries) {
            return false;
        }

        // Check entry requirements
        switch ($this->entry_requirement) {
            case 'membership':
                return $user->hasActiveMembership();
            case 'purchase':
                return $user->purchases()->where('status', 'completed')->exists();
            case 'both':
                return $user->hasActiveMembership() 
                    && $user->purchases()->where('status', 'completed')->exists();
            case 'none':
            default:
                return true;
        }
    }

    public function selectWinner()
    {
        $entries = $this->entries()->get();
        
        if ($entries->isEmpty()) {
            return null;
        }

        // Create weighted array based on entries_count (VIP multiplier)
        $weightedEntries = [];
        foreach ($entries as $entry) {
            for ($i = 0; $i < $entry->entries_count; $i++) {
                $weightedEntries[] = $entry->user_id;
            }
        }

        // Randomly select winner
        $winnerUserId = $weightedEntries[array_rand($weightedEntries)];

        $this->winner_user_id = $winnerUserId;
        $this->winner_selected_at = now();
        $this->status = 'ended';
        $this->save();

        return $this->winner;
    }
}
