<?php

namespace App\Jobs;

use App\Mail\GiveawayWinnerEmail;
use App\Models\Giveaway;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;

class ProcessEndedGiveawaysJob implements ShouldQueue
{
    use Queueable;

    /**
     * The number of times the job may be attempted.
     *
     * @var int
     */
    public $tries = 3;

    /**
     * The number of seconds the job can run before timing out.
     *
     * @var int
     */
    public $timeout = 300;

    /**
     * Create a new job instance.
     */
    public function __construct()
    {
        //
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        // Find all active giveaways that have ended but don't have a winner yet
        $endedGiveaways = Giveaway::where('status', 'active')
            ->where('end_date', '<', now())
            ->whereNull('winner_user_id')
            ->get();

        $count = 0;
        foreach ($endedGiveaways as $giveaway) {
            // Update status to ended
            $giveaway->update(['status' => 'ended']);

            // Automatically select a winner if there are entries
            if ($giveaway->total_entries > 0) {
                $winner = $giveaway->selectWinner();
                
                if ($winner) {
                    // Send winner notification email
                    Mail::to($winner->email)->send(new GiveawayWinnerEmail($giveaway, $winner));
                    $count++;
                    Log::info("ProcessEndedGiveawaysJob: Winner selected for giveaway #{$giveaway->id}: {$winner->name}");
                }
            }
        }

        Log::info("ProcessEndedGiveawaysJob: Processed {$count} ended giveaways and selected winners");
    }
}
