<?php

namespace App\Http\Controllers;

use App\Models\MembershipTier;
use App\Models\MembershipPurchase;
use App\Models\MembershipCard;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;

class MembershipController extends Controller
{
    public function index()
    {
        $tiers = MembershipTier::active()->ordered()->get();
        $activeMembership = auth()->user()->getActiveMembershipCard();
        
        return view('memberships.index', compact('tiers', 'activeMembership'));
    }

    public function purchase(Request $request, MembershipTier $membershipTier)
    {
        $user = auth()->user();

        // Check if tier is active
        if (!$membershipTier->is_active) {
            return redirect()->back()
                            ->with('error', 'This membership tier is not available.');
        }

        // Get user's current active membership if any
        $activeMembership = $user->getActiveMembershipCard();
        
        if ($activeMembership) {
            $currentTier = $activeMembership->membershipTier;
            $newTier = $membershipTier;
            
            // Allow same tier (renewal/extension)
            if ($currentTier->id === $newTier->id) {
                // This is a renewal - allow it
            }
            // Allow upgrade to higher tier (compare by order or price)
            elseif ($newTier->order < $currentTier->order || $newTier->price > $currentTier->price) {
                // This is an upgrade - allow it
                // Mark current membership as cancelled since they're upgrading
                $activeMembership->update(['status' => 'cancelled']);
            }
            // Block downgrade to lower tier
            else {
                return redirect()->back()
                                ->with('error', 'You cannot downgrade to a lower tier while you have an active membership. Please wait for your current membership to expire.');
            }
        }

        try {
            DB::beginTransaction();

            // Create purchase record
            $purchase = MembershipPurchase::create([
                'user_id' => $user->id,
                'membership_tier_id' => $membershipTier->id,
                'amount' => $membershipTier->price,
                'currency' => $membershipTier->currency,
                'payment_method' => null,
                'status' => 'pending',
            ]);

            DB::commit();

            // Redirect to deposit page with purchase info
            return redirect()->route('memberships.pending', $purchase)
                            ->with('success', 'Please complete the payment to activate your membership.');

        } catch (\Exception $e) {
            DB::rollBack();
            
            return redirect()->back()
                            ->with('error', $e->getMessage());
        }
    }

    public function pending(MembershipPurchase $membershipPurchase)
    {
        // Ensure user owns this purchase
        if ($membershipPurchase->user_id !== auth()->id()) {
            abort(403);
        }

        return view('memberships.pending', compact('membershipPurchase'));
    }
}
