<?php

namespace App\Http\Controllers;

use App\Models\Giveaway;
use App\Models\GiveawayEntry;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class GiveawayController extends Controller
{
    public function index()
    {
        $activeGiveaways = Giveaway::active()
                                   ->with(['prizeCar', 'entries'])
                                   ->latest('end_date')
                                   ->get();

        $upcomingGiveaways = Giveaway::upcoming()
                                     ->with(['prizeCar'])
                                     ->latest('start_date')
                                     ->get();

        $user = auth()->user();
        $userEntries = $user->giveawayEntries()->pluck('giveaway_id')->toArray();

        return view('giveaways.index', compact('activeGiveaways', 'upcomingGiveaways', 'userEntries'));
    }

    public function show(Giveaway $giveaway)
    {
        $giveaway->load(['prizeCar', 'winner', 'entries']);
        
        $user = auth()->user();
        $hasEntered = $user->hasEnteredGiveaway($giveaway->id);
        $userEntry = $user->giveawayEntries()->where('giveaway_id', $giveaway->id)->first();
        $canEnter = $giveaway->canUserEnter($user);
        
        return view('giveaways.show', compact('giveaway', 'hasEntered', 'userEntry', 'canEnter'));
    }

    public function enter(Giveaway $giveaway)
    {
        $user = auth()->user();

        // Check if user can enter
        if (!$giveaway->canUserEnter($user)) {
            return redirect()->back()
                            ->with('error', 'You are not eligible to enter this giveaway.');
        }

        try {
            DB::beginTransaction();

            // Create entry
            $entry = GiveawayEntry::create([
                'giveaway_id' => $giveaway->id,
                'user_id' => $user->id,
                'entries_count' => 1, // Will be multiplied by VIP tier in model boot
                'entry_source' => 'manual',
                'entered_at' => now(),
            ]);

            DB::commit();

            $multiplier = $entry->entries_count;
            $message = $multiplier > 1 
                ? "You've successfully entered the giveaway with {$multiplier}x entries (VIP bonus)!" 
                : "You've successfully entered the giveaway!";

            return redirect()->back()
                            ->with('success', $message);

        } catch (\Exception $e) {
            DB::rollBack();
            
            return redirect()->back()
                            ->with('error', 'Failed to enter giveaway: ' . $e->getMessage());
        }
    }

    public function myEntries()
    {
        $user = auth()->user();
        $entries = $user->giveawayEntries()
                       ->with(['giveaway.prizeCar'])
                       ->latest()
                       ->paginate(15);

        return view('giveaways.my-entries', compact('entries'));
    }
}
