<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Cache;

class SettingController extends Controller
{
    public function index()
    {
        $settings = Setting::getGrouped();
        
        // Ensure required groups exist (even if empty) to prevent undefined array key errors
        $requiredGroups = ['general', 'security', 'appearance', 'communication', 'livechat'];
        foreach ($requiredGroups as $group) {
            if (!isset($settings[$group])) {
                $settings[$group] = collect();
            }
        }
        
        return view('admin.settings.index', compact('settings'));
    }

    public function update(Request $request)
    {
        // Build validation rules
        $validationRules = [
            'settings' => 'required|array',
            'settings.*' => 'nullable|string',
        ];

        // Add file validation rules for image settings
        $imageSettings = Setting::where('type', 'image')->get();
        foreach ($imageSettings as $setting) {
            $validationRules["file_{$setting->key}"] = 'nullable|file|mimes:jpeg,png,jpg,gif,svg,ico|max:2048';
        }

        $request->validate($validationRules);

        try {
            
            // Iterate over ALL settings so image-only fields are handled
            $allSettings = Setting::all();
            foreach ($allSettings as $setting) {
                $key = $setting->key;
                $value = $request->input("settings.$key", $setting->value);

                // Handle file uploads for image settings
                if ($setting->isImage()) {
                    
                    
                    if ($request->hasFile("file_{$key}")) {
                        $file = $request->file("file_{$key}");
                        
                        // Delete old file if exists
                        if ($setting->value && Storage::disk('public_uploads')->exists($setting->value)) {
                            Storage::disk('public_uploads')->delete($setting->value);
                        }

                        // Store new file
                        $path = $file->store('settings', 'public_uploads');
                        $value = $path;
                        
                    }
                }

                // Handle checkbox values
                if ($setting->isCheckbox()) {
                    $value = $request->boolean("settings.$key") ? '1' : '0';
                }

                $setting->update(['value' => $value]);
            }

            // Clear settings cache
            Setting::clearCache();

            return redirect()->route('admin.settings.index')
                ->with('success', 'Settings updated successfully.');

        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to update settings. Please try again.')
                ->withInput();
        }
    }

    public function clearCache()
    {
        try {
            Setting::clearCache();
            Cache::flush();
            
            return redirect()->route('admin.settings.index')
                ->with('success', 'Cache cleared successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to clear cache. Please try again.');
        }
    }

    public function resetToDefaults()
    {
        try {
            // Delete all current settings
            Setting::truncate();
            
            // Run the seeder to restore defaults
            \Artisan::call('db:seed', ['--class' => 'SettingsSeeder']);
            
            // Clear cache
            Setting::clearCache();
            
            return redirect()->route('admin.settings.index')
                ->with('success', 'Settings reset to defaults successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to reset settings. Please try again.');
        }
    }

    public function fixStorage()
    {
        try {
            $messages = [];
            $publicStoragePath = public_path('storage');
            $storageAppPublicPath = storage_path('app/public');
            
            // Check if public/storage exists and remove it
            if (file_exists($publicStoragePath)) {
                if (is_link($publicStoragePath)) {
                    @unlink($publicStoragePath);
                    $messages[] = 'Removed existing symbolic link.';
                } elseif (is_dir($publicStoragePath)) {
                    // It's a directory, not a symlink - remove it
                    @rmdir($publicStoragePath);
                    $messages[] = 'Removed existing directory.';
                }
            }
            
            // Create .htaccess files for storage access
            $htaccessContent = "<IfModule mod_rewrite.c>\n    RewriteEngine Off\n</IfModule>\n\n";
            $htaccessContent .= "<IfModule mod_autoindex.c>\n    Options -Indexes\n</IfModule>\n\n";
            $htaccessContent .= "# Allow access to all files - Apache 2.4 syntax\n";
            $htaccessContent .= "<IfModule mod_authz_core.c>\n    Require all granted\n</IfModule>\n\n";
            $htaccessContent .= "# Fallback for Apache 2.2\n";
            $htaccessContent .= "<IfModule !mod_authz_core.c>\n    Order allow,deny\n    Allow from all\n</IfModule>\n\n";
            $htaccessContent .= "# Set proper MIME types for images\n";
            $htaccessContent .= "<IfModule mod_mime.c>\n";
            $htaccessContent .= "    AddType image/jpeg .jpg .jpeg\n";
            $htaccessContent .= "    AddType image/png .png\n";
            $htaccessContent .= "    AddType image/gif .gif\n";
            $htaccessContent .= "    AddType image/svg+xml .svg\n";
            $htaccessContent .= "    AddType image/webp .webp\n";
            $htaccessContent .= "</IfModule>\n";
            
            // Create .htaccess in storage/app/public
            file_put_contents($storageAppPublicPath . '/.htaccess', $htaccessContent);
            $messages[] = 'Created .htaccess in storage/app/public.';
            
            // Try to create symlink (works on Linux/Mac)
            if (strtoupper(substr(PHP_OS, 0, 3)) !== 'WIN') {
                symlink($storageAppPublicPath, $publicStoragePath);
                $messages[] = 'Created symbolic link.';
            } else {
                // For Windows, try junction
                $cmd = 'mklink /J "' . $publicStoragePath . '" "' . $storageAppPublicPath . '"';
                exec($cmd, $output, $returnCode);
                
                if ($returnCode === 0) {
                    $messages[] = 'Created storage junction (Windows).';
                } else {
                    // Fallback: try Laravel's storage:link command
                    \Artisan::call('storage:link');
                    $messages[] = 'Created storage link via Artisan.';
                }
            }
            
            // Create .htaccess in public/storage
            if (file_exists($publicStoragePath)) {
                file_put_contents($publicStoragePath . '/.htaccess', $htaccessContent);
                $messages[] = 'Created .htaccess in public/storage.';
            }
            
            // Set permissions (Unix-like systems)
            if (strtoupper(substr(PHP_OS, 0, 3)) !== 'WIN') {
                @chmod($storageAppPublicPath, 0755);
                @chmod($publicStoragePath, 0755);
                $messages[] = 'Set proper permissions.';
            }
            
            // Clear cache
            Setting::clearCache();
            Cache::flush();
            
            $successMessage = 'Storage access fixed successfully! ' . implode(' ', $messages);
            
            return redirect()->route('admin.settings.index')
                ->with('success', $successMessage);
                
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to fix storage access: ' . $e->getMessage());
        }
    }

    public function storageLink()
    {
        try {
            \Artisan::call('storage:link');
            
            return redirect()->route('admin.settings.index')
                ->with('success', 'Storage link created successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to create storage link: ' . $e->getMessage());
        }
    }

    public function clearConfig()
    {
        try {
            \Artisan::call('config:clear');
            
            return redirect()->route('admin.settings.index')
                ->with('success', 'Configuration cache cleared successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to clear config cache.');
        }
    }

    public function clearViews()
    {
        try {
            \Artisan::call('view:clear');
            
            return redirect()->route('admin.settings.index')
                ->with('success', 'Compiled views cleared successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to clear views.');
        }
    }

    public function clearRoutes()
    {
        try {
            \Artisan::call('route:clear');
            
            return redirect()->route('admin.settings.index')
                ->with('success', 'Route cache cleared successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to clear routes.');
        }
    }

    public function optimizeClear()
    {
        try {
            \Artisan::call('optimize:clear');
            Setting::clearCache();
            Cache::flush();
            
            return redirect()->route('admin.settings.index')
                ->with('success', 'All caches cleared successfully (config, views, routes, cache, compiled).');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to clear all caches.');
        }
    }
}
