<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\MembershipTier;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class MembershipTierController extends Controller
{
    public function index()
    {
        $tiers = MembershipTier::ordered()->paginate(15);
        
        return view('admin.membership-tiers.index', compact('tiers'));
    }

    public function create()
    {
        return view('admin.membership-tiers.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'price' => 'required|numeric|min:0',
            'duration_days' => 'required|integer|min:1',
            'benefits' => 'nullable|string',
            'car_discount_percentage' => 'nullable|numeric|min:0|max:100',
            'investment_bonus_percentage' => 'nullable|numeric|min:0|max:100',
            'giveaway_entry_multiplier' => 'nullable|integer|min:1',
            'priority_support' => 'boolean',
            'is_active' => 'boolean',
            'sort_order' => 'nullable|integer',
        ]);

        // Convert benefits string to array
        if (isset($validated['benefits'])) {
            $benefits = array_filter(array_map('trim', explode("\n", $validated['benefits'])));
            $validated['benefits'] = $benefits;
        }

        $validated['slug'] = Str::slug($validated['name']);
        $validated['currency'] = auth()->user()->currency ?? 'USD';

        MembershipTier::create($validated);

        return redirect()->route('admin.membership-tiers.index')
                        ->with('success', 'Membership tier created successfully.');
    }

    public function edit(MembershipTier $membershipTier)
    {
        return view('admin.membership-tiers.edit', compact('membershipTier'));
    }

    public function update(Request $request, MembershipTier $membershipTier)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'price' => 'required|numeric|min:0',
            'duration_days' => 'required|integer|min:1',
            'benefits' => 'nullable|string',
            'car_discount_percentage' => 'nullable|numeric|min:0|max:100',
            'investment_bonus_percentage' => 'nullable|numeric|min:0|max:100',
            'giveaway_entry_multiplier' => 'nullable|integer|min:1',
            'priority_support' => 'boolean',
            'is_active' => 'boolean',
            'sort_order' => 'nullable|integer',
        ]);

        // Convert benefits string to array
        if (isset($validated['benefits'])) {
            $benefits = array_filter(array_map('trim', explode("\n", $validated['benefits'])));
            $validated['benefits'] = $benefits;
        }

        $validated['slug'] = Str::slug($validated['name']);

        $membershipTier->update($validated);

        return redirect()->route('admin.membership-tiers.index')
                        ->with('success', 'Membership tier updated successfully.');
    }

    public function destroy(MembershipTier $membershipTier)
    {
        // Check if tier has active cards
        if ($membershipTier->cards()->where('status', 'active')->exists()) {
            return redirect()->route('admin.membership-tiers.index')
                            ->with('error', 'Cannot delete tier with active membership cards.');
        }

        $membershipTier->delete();

        return redirect()->route('admin.membership-tiers.index')
                        ->with('success', 'Membership tier deleted successfully.');
    }

    public function toggle(MembershipTier $membershipTier)
    {
        $membershipTier->update(['is_active' => !$membershipTier->is_active]);

        return redirect()->route('admin.membership-tiers.index')
                        ->with('success', 'Membership tier status updated successfully.');
    }
}
