<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\MembershipPurchase;
use App\Models\User;
use App\Models\MembershipTier;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\DB;

class MembershipPurchaseController extends Controller
{
    public function index(Request $request)
    {
        $query = MembershipPurchase::with(['user', 'membershipTier', 'membershipCard'])
                                   ->latest();

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filter by tier
        if ($request->filled('tier')) {
            $query->where('membership_tier_id', $request->tier);
        }

        // Search by user or transaction ID
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('transaction_id', 'like', "%{$search}%")
                  ->orWhereHas('user', function($q) use ($search) {
                      $q->where('name', 'like', "%{$search}%")
                        ->orWhere('email', 'like', "%{$search}%");
                  });
            });
        }

        $purchases = $query->paginate(20);
        $tiers = MembershipTier::active()->ordered()->get();

        return view('admin.membership-purchases.index', compact('purchases', 'tiers'));
    }

    public function show(MembershipPurchase $membershipPurchase)
    {
        $membershipPurchase->load(['user', 'membershipTier', 'membershipCard']);
        
        return view('admin.membership-purchases.show', compact('membershipPurchase'));
    }

    public function approve(MembershipPurchase $membershipPurchase)
    {
        if ($membershipPurchase->status !== 'pending') {
            return redirect()->back()
                            ->with('error', 'Only pending purchases can be approved.');
        }

        try {
            DB::beginTransaction();

            // Mark purchase as completed
            $membershipPurchase->markAsCompleted();

            // Create membership card
            $tier = $membershipPurchase->membershipTier;
            $expiresAt = now()->addDays($tier->duration_days);
            
            $card = \App\Models\MembershipCard::create([
                'user_id' => $membershipPurchase->user_id,
                'membership_tier_id' => $tier->id,
                'issued_at' => now(),
                'expires_at' => $expiresAt,
                'status' => 'active',
            ]);

            // Link card to purchase
            $membershipPurchase->update(['membership_card_id' => $card->id]);

            // Generate QR code
            $card->generateQrCode();

            // Send confirmation email
            Mail::to($membershipPurchase->user->email)->send(new \App\Mail\MembershipPurchasedEmail($membershipPurchase));

            DB::commit();

            return redirect()->back()
                            ->with('success', 'Membership purchase approved and card activated successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            
            return redirect()->back()
                            ->with('error', 'Failed to approve purchase: ' . $e->getMessage());
        }
    }

    public function reject(Request $request, MembershipPurchase $membershipPurchase)
    {
        if ($membershipPurchase->status !== 'pending') {
            return redirect()->back()
                            ->with('error', 'Only pending purchases can be rejected.');
        }

        $notes = $request->input('notes', 'Rejected by admin');
        $membershipPurchase->markAsFailed($notes);

        return redirect()->back()
                        ->with('success', 'Membership purchase rejected.');
    }
}
