<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Giveaway;
use App\Models\Car;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Mail;

class GiveawayController extends Controller
{
    public function index(Request $request)
    {
        $query = Giveaway::with(['prizeCar', 'winner'])->latest();

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        $giveaways = $query->paginate(15);
        
        return view('admin.giveaways.index', compact('giveaways'));
    }

    public function create()
    {
        $cars = Car::where('is_available', true)->get();
        
        return view('admin.giveaways.create', compact('cars'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'prize_type' => 'required|in:car,cash,investment,other',
            'prize_value' => 'nullable|numeric|min:0',
            'prize_car_id' => 'nullable|exists:cars,id',
            'prize_details' => 'nullable|string',
            'max_entries' => 'nullable|integer|min:1',
            'entry_requirement' => 'required|in:none,membership,purchase,both',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
            'status' => 'required|in:draft,active,ended,cancelled',
            'is_featured' => 'boolean',
        ]);

        // Handle image upload
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $filename = time() . '_' . $image->getClientOriginalName();
            $image->move(public_path('uploads/giveaways'), $filename);
            $validated['image_path'] = 'giveaways/' . $filename;
        }

        $validated['currency'] = auth()->user()->currency ?? 'USD';

        $giveaway = Giveaway::create($validated);

        return redirect()->route('admin.giveaways.index')
                        ->with('success', 'Giveaway created successfully.');
    }

    public function edit(Giveaway $giveaway)
    {
        $cars = Car::where('is_available', true)->get();
        
        return view('admin.giveaways.edit', compact('giveaway', 'cars'));
    }

    public function update(Request $request, Giveaway $giveaway)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'prize_type' => 'required|in:car,cash,investment,other',
            'prize_value' => 'nullable|numeric|min:0',
            'prize_car_id' => 'nullable|exists:cars,id',
            'prize_details' => 'nullable|string',
            'max_entries' => 'nullable|integer|min:1',
            'entry_requirement' => 'required|in:none,membership,purchase,both',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
            'status' => 'required|in:draft,active,ended,cancelled',
            'is_featured' => 'boolean',
        ]);

        // Handle image upload
        if ($request->hasFile('image')) {
            // Delete old image
            if ($giveaway->image_path && file_exists(public_path('uploads/' . $giveaway->image_path))) {
                unlink(public_path('uploads/' . $giveaway->image_path));
            }

            $image = $request->file('image');
            $filename = time() . '_' . $image->getClientOriginalName();
            $image->move(public_path('uploads/giveaways'), $filename);
            $validated['image_path'] = 'giveaways/' . $filename;
        }

        $giveaway->update($validated);

        return redirect()->route('admin.giveaways.index')
                        ->with('success', 'Giveaway updated successfully.');
    }

    public function destroy(Giveaway $giveaway)
    {
        // Delete image if exists
        if ($giveaway->image_path && file_exists(public_path('uploads/' . $giveaway->image_path))) {
            unlink(public_path('uploads/' . $giveaway->image_path));
        }

        $giveaway->delete();

        return redirect()->route('admin.giveaways.index')
                        ->with('success', 'Giveaway deleted successfully.');
    }

    public function show(Giveaway $giveaway)
    {
        $giveaway->load(['prizeCar', 'winner', 'entries.user']);
        $entries = $giveaway->entries()->with('user')->latest()->paginate(20);
        
        return view('admin.giveaways.show', compact('giveaway', 'entries'));
    }

    public function selectWinner(Giveaway $giveaway)
    {
        if ($giveaway->has_winner) {
            return redirect()->back()
                            ->with('error', 'Winner has already been selected for this giveaway.');
        }

        if ($giveaway->entries()->count() === 0) {
            return redirect()->back()
                            ->with('error', 'No entries found for this giveaway.');
        }

        $winner = $giveaway->selectWinner();

        // Send winner notification email
        Mail::to($winner->email)->send(new \App\Mail\GiveawayWinnerEmail($giveaway, $winner));

        return redirect()->back()
                        ->with('success', 'Winner selected successfully: ' . $winner->name);
    }
}
