<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\DB;

class MigrateImagesToPublic extends Command
{
    protected $signature = 'images:migrate-to-public {--force : Force migration even if uploads directory already has files}';
    protected $description = 'Migrate images from storage/app/public to public/uploads for shared hosting compatibility';

    public function handle()
    {
        $this->info('╔════════════════════════════════════════════════════╗');
        $this->info('║   Tesla Platform - Image Migration to Public      ║');
        $this->info('╚════════════════════════════════════════════════════╝');
        $this->newLine();
        
        $sourcePath = storage_path('app/public');
        $destPath = public_path('uploads');
        
        // Check if source exists
        if (!File::exists($sourcePath)) {
            $this->warn('Source directory storage/app/public does not exist.');
            $this->info('This might be a fresh installation. Nothing to migrate.');
            return 0;
        }
        
        // Check if uploads directory exists and has files
        if (File::exists($destPath) && !$this->option('force')) {
            $filesCount = count(File::allFiles($destPath));
            if ($filesCount > 0) {
                $this->warn("Uploads directory already contains {$filesCount} files.");
                if (!$this->confirm('Do you want to continue? This will overwrite existing files.', false)) {
                    $this->info('Migration cancelled.');
                    return 0;
                }
            }
        }
        
        // Create uploads directory if it doesn't exist
        if (!File::exists($destPath)) {
            File::makeDirectory($destPath, 0755, true);
            $this->info('✓ Created uploads directory');
        }
        
        // Copy files from storage to public
        $this->info('Copying files from storage/app/public to public/uploads...');
        $bar = $this->output->createProgressBar(100);
        $bar->start();
        
        try {
            // Get subdirectories
            $directories = ['profile-images', 'cars', 'settings', 'payment-methods', 'kyc', 'investment-plans'];
            $copiedFiles = 0;
            $copiedDirs = 0;
            
            foreach ($directories as $dir) {
                $sourceDir = $sourcePath . '/' . $dir;
                $destDir = $destPath . '/' . $dir;
                
                if (File::exists($sourceDir)) {
                    File::copyDirectory($sourceDir, $destDir);
                    $filesInDir = count(File::allFiles($destDir));
                    $copiedFiles += $filesInDir;
                    $copiedDirs++;
                    $bar->advance(10);
                }
            }
            
            $bar->finish();
            $this->newLine(2);
            $this->info("✓ Copied {$copiedDirs} directories with {$copiedFiles} files");
            
        } catch (\Exception $e) {
            $this->newLine(2);
            $this->error('Failed to copy files: ' . $e->getMessage());
            return 1;
        }
        
        // Update database paths - paths remain the same, just the disk reference changes
        $this->info('Verifying database records...');
        
        try {
            // Count records with images
            $usersCount = DB::table('users')->whereNotNull('profile_image')->count();
            $carsCount = DB::table('cars')->whereNotNull('images')->count();
            $settingsCount = DB::table('settings')->where('type', 'image')->whereNotNull('value')->count();
            $paymentMethodsCount = DB::table('payment_methods')
                ->where(function($q) {
                    $q->whereNotNull('logo')->orWhereNotNull('barcode');
                })
                ->count();
            $kycsCount = DB::table('kycs')
                ->where(function($q) {
                    $q->whereNotNull('document_front_path')
                      ->orWhereNotNull('document_back_path')
                      ->orWhereNotNull('selfie_path');
                })
                ->count();
            $plansCount = DB::table('investment_plans')->whereNotNull('image')->count();
            
            $this->info("Database records found:");
            $this->info("  • Users with profile images: {$usersCount}");
            $this->info("  • Cars with images: {$carsCount}");
            $this->info("  • Settings with images: {$settingsCount}");
            $this->info("  • Payment methods with images: {$paymentMethodsCount}");
            $this->info("  • KYC records with documents: {$kycsCount}");
            $this->info("  • Investment plans with images: {$plansCount}");
            
            $this->newLine();
            $this->info('✓ Database paths are compatible (no changes needed)');
            
        } catch (\Exception $e) {
            $this->warn('Could not verify database: ' . $e->getMessage());
        }
        
        // Create .htaccess for proper MIME types
        $this->info('Creating .htaccess for proper MIME types...');
        $htaccessContent = <<<'HTACCESS'
<IfModule mod_rewrite.c>
    RewriteEngine Off
</IfModule>

<IfModule mod_autoindex.c>
    Options -Indexes
</IfModule>

# Allow access to all files - Apache 2.4 syntax
<IfModule mod_authz_core.c>
    Require all granted
</IfModule>

# Fallback for Apache 2.2
<IfModule !mod_authz_core.c>
    Order allow,deny
    Allow from all
</IfModule>

# Set proper MIME types for images
<IfModule mod_mime.c>
    AddType image/jpeg .jpg .jpeg
    AddType image/png .png
    AddType image/gif .gif
    AddType image/svg+xml .svg
    AddType image/webp .webp
    AddType image/x-icon .ico
</IfModule>

# Enable caching for better performance
<IfModule mod_expires.c>
    ExpiresActive On
    ExpiresByType image/jpeg "access plus 1 year"
    ExpiresByType image/png "access plus 1 year"
    ExpiresByType image/gif "access plus 1 year"
    ExpiresByType image/svg+xml "access plus 1 year"
</IfModule>
HTACCESS;
        
        File::put($destPath . '/.htaccess', $htaccessContent);
        $this->info('✓ Created .htaccess in public/uploads');
        
        $this->newLine();
        $this->info('╔════════════════════════════════════════════════════╗');
        $this->info('║           Migration Completed Successfully!       ║');
        $this->info('╚════════════════════════════════════════════════════╝');
        $this->newLine();
        
        $this->comment('Next steps:');
        $this->comment('1. Test image uploads and display on all features');
        $this->comment('2. Verify both admin and user views work correctly');
        $this->comment('3. After confirming everything works:');
        $this->comment('   php artisan images:cleanup-old-storage');
        $this->newLine();
        
        return 0;
    }
}
