<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\File;

class CleanupOldStorage extends Command
{
    protected $signature = 'images:cleanup-old-storage {--force : Skip confirmation}';
    protected $description = 'Remove old image files from storage/app/public after migration';

    public function handle()
    {
        $storagePath = storage_path('app/public');
        
        if (!File::exists($storagePath)) {
            $this->info('Storage directory does not exist. Nothing to clean up.');
            return 0;
        }
        
        // Get directories to clean
        $directories = ['profile-images', 'cars', 'settings', 'payment-methods', 'kyc', 'investment-plans'];
        $filesToDelete = 0;
        
        foreach ($directories as $dir) {
            $path = $storagePath . '/' . $dir;
            if (File::exists($path)) {
                $filesToDelete += count(File::allFiles($path));
            }
        }
        
        if ($filesToDelete === 0) {
            $this->info('No files to clean up.');
            return 0;
        }
        
        $this->warn("This will delete {$filesToDelete} files from storage/app/public");
        $this->warn('Make sure you have verified that images work correctly in public/uploads!');
        $this->newLine();
        
        if (!$this->option('force')) {
            if (!$this->confirm('Are you sure you want to delete these files?', false)) {
                $this->info('Cleanup cancelled.');
                return 0;
            }
        }
        
        $this->info('Deleting old storage files...');
        $bar = $this->output->createProgressBar(count($directories));
        $bar->start();
        
        $deletedFiles = 0;
        $deletedDirs = 0;
        
        foreach ($directories as $dir) {
            $path = $storagePath . '/' . $dir;
            if (File::exists($path)) {
                $deletedFiles += count(File::allFiles($path));
                File::deleteDirectory($path);
                $deletedDirs++;
            }
            $bar->advance();
        }
        
        $bar->finish();
        $this->newLine(2);
        
        $this->info("✓ Deleted {$deletedDirs} directories with {$deletedFiles} files");
        $this->info('✓ Cleanup completed successfully!');
        
        return 0;
    }
}
