# Cron Job Setup for Tesla Drives Investment Platform

## Overview
This guide explains how to set up cron jobs for the Tesla Drives investment platform with **intelligent caching** and **Finnhub API rate limiting** to prevent API quota exhaustion.

## 🚀 New Intelligent Caching System

The system now includes intelligent caching that prevents redundant API calls and respects Finnhub's 30 calls/second rate limit:

- **Stock Updates**: Run every 15 minutes (cached)
- **Investment Monitoring**: Run every 30 minutes (cached)  
- **Automatic Investments**: Run every 60 minutes (cached)
- **Stock News**: Run every 6 hours (cached)
- **Data Cleanup**: Run every 24 hours (cached)
 cron job command
*/5 * * * * wget -q -O- "https://yoururl/cron/run-all?token=73530" >/dev/null 2>&1
if the above command doesn't work for you use commands below
## Recommended Cron Jobs

### 1. 🎯 All-in-One Smart Scheduler (Recommended: Every 10 minutes)
```bash
*/10 * * * * cd /path/to/your/project && php artisan cron:queue-manager run-all
```
**This single cron job handles everything with intelligent caching!**

### 2. 📊 System Monitoring (Optional: Every 5 minutes)
```bash
*/5 * * * * cd /path/to/your/project && php artisan cron:queue-manager monitor
```

## Alternative Individual Cron Jobs (Legacy)

If you prefer individual cron jobs instead of the smart scheduler:

### Stock Updates (Every 15 minutes during market hours)
```bash
*/15 9-16 * * 1-5 cd /path/to/your/project && php artisan cron:queue-manager dispatch
```

### Automatic Investments (Daily at 9 AM)
```bash
0 9 * * * cd /path/to/your/project && php artisan cron:queue-manager investments
```

### Data Cleanup (Weekly on Sunday at 2 AM)
```bash
0 2 * * 0 cd /path/to/your/project && php artisan cron:queue-manager cleanup
```

## 🔧 Cron Job Actions

### 🎯 Run-All Action (`run-all`) - **RECOMMENDED**
**The intelligent all-in-one scheduler with caching:**
- **Stock & Prices**: Updates stock quotes, monitors prices, monitors investments (every 15 min)
- **Automatic Investments**: Executes due investment plans (every 60 min)
- **Stock News**: Processes news articles with sentiment analysis (every 6 hours)
- **Data Cleanup**: Removes old data and logs (every 24 hours)
- **Cache Status**: Shows when each task last ran and next scheduled run
- **Rate Limiting**: Respects Finnhub's 30 calls/sec limit with 2-second delays
- **Smart Execution**: Only runs tasks when needed, skips recent executions

### 📊 Monitor Action (`monitor`)
**System health monitoring with individual task caching:**
- Checks queue worker status (sync mode)
- Monitors automatic investment plans (cached 60 min)
- Monitors stock prices (cached 30 min)  
- Monitors investment prices (cached 30 min)
- Processes stock news (cached 6 hours)
- Retries failed jobs and shows system status

### 💰 Investments Action (`investments`)
**Direct automatic investment execution:**
- Executes all due automatic investment plans
- Validates wallet balances before investing
- Creates investment holdings and transactions
- Updates next investment dates
- Comprehensive error handling and logging

### 📈 Stock-Monitor Action (`stock-monitor`)
**Real-time stock monitoring:**
- Monitors user stock holdings for price changes
- Creates notifications for significant changes (≥2%)
- Checks price alerts for watchlist items
- Updates holding values and gain/loss calculations

### 🔄 Dispatch Action (`dispatch`)
**Stock data updates with rate limiting:**
- Updates quotes for all active stocks
- Implements 2-second delays between API calls
- Respects Finnhub's 30 calls/second rate limit
- Comprehensive error handling for API failures

### 🧹 Cleanup Action (`cleanup`)
**Data maintenance:**
- Removes old stock quotes and price history
- Cleans up old news articles and API logs
- Maintains database performance
- Configurable retention periods

## 🛠️ Setup Instructions

### Option 1: Smart Scheduler (Recommended)

1. **Access your server's cron configuration:**
   ```bash
   crontab -e
   ```

2. **Add the smart scheduler cron job:**
   ```bash
   # Tesla Investment Platform - Smart Scheduler with Intelligent Caching
   */10 * * * * cd /home/cycloverseworld/domains/themevortex.com/public_html && php artisan cron:queue-manager run-all >> /tmp/tesla-cron.log 2>&1
   
   # Optional: System monitoring
   */5 * * * * cd /home/cycloverseworld/domains/themevortex.com/public_html && php artisan cron:queue-manager monitor >> /tmp/tesla-monitor.log 2>&1
   ```

### Option 2: Individual Cron Jobs (Legacy)

If you prefer individual control over each task:

```bash
# Automatic investments - daily at 9 AM
0 9 * * * cd /home/cycloverseworld/domains/themevortex.com/public_html && php artisan cron:queue-manager investments

# Stock updates - every 15 minutes during market hours (with rate limiting)
*/15 9-16 * * 1-5 cd /home/cycloverseworld/domains/themevortex.com/public_html && php artisan cron:queue-manager dispatch

# Stock monitoring - every 30 minutes during market hours
*/30 9-16 * * 1-5 cd /home/cycloverseworld/domains/themevortex.com/public_html && php artisan cron:queue-manager stock-monitor

# News processing - every 6 hours
0 */6 * * * cd /home/cycloverseworld/domains/themevortex.com/public_html && php artisan cron:queue-manager news

# Data cleanup - weekly on Sunday at 2 AM
0 2 * * 0 cd /home/cycloverseworld/domains/themevortex.com/public_html && php artisan cron:queue-manager cleanup
```

3. **Save and exit** the cron editor (Ctrl+X, then Y, then Enter)

4. **Verify the cron jobs are active:**
   ```bash
   crontab -l
   ```

## 🧪 Testing

### Test the Smart Scheduler
```bash
# Test the intelligent caching system
php artisan cron:queue-manager run-all

# Run it again immediately to see caching in action
php artisan cron:queue-manager run-all
```

### Test Individual Actions
```bash
# Test system monitoring with caching
php artisan cron:queue-manager monitor

# Test automatic investments
php artisan cron:queue-manager investments

# Test stock updates (with rate limiting)
php artisan cron:queue-manager dispatch

# Test stock price monitoring
php artisan cron:queue-manager stock-monitor

# Test news processing
php artisan cron:queue-manager news

# Test data cleanup
php artisan cron:queue-manager cleanup
```

## 📊 Monitoring & Logs

### Application Logs
```bash
# Watch Laravel logs in real-time
tail -f storage/logs/laravel.log

# Check for API rate limit errors
grep "API limit reached" storage/logs/laravel.log

# Check caching activity
grep "Cache Status" storage/logs/laravel.log
```

### Cron Logs
```bash
# Check smart scheduler logs
tail -f /tmp/tesla-cron.log

# Check monitoring logs
tail -f /tmp/tesla-monitor.log

# System cron logs
sudo tail -f /var/log/cron
```

### Cache Status
The smart scheduler shows cache status for each task:
- ✅ **Ready to run**: Task hasn't run recently
- ⏳ **Next in Xmin**: Task cached, shows time until next run
- 🔄 **Running**: Task currently executing
- ✅ **Completed in Xms**: Task finished successfully

## 💰 Automatic Investment Features

The intelligent investment system includes:

- **Smart Scheduling**: Cached execution prevents duplicate investments
- **Wallet Validation**: Checks sufficient funds before investment
- **Transaction Creation**: Creates proper wallet and investment transactions
- **Holding Updates**: Updates or creates investment holdings
- **Next Date Calculation**: Automatically calculates next investment dates
- **Error Handling**: Comprehensive error logging and handling
- **Rate Limiting**: Respects API limits when fetching investment data

## 🔒 Security & Performance Notes

- **API Rate Limiting**: Finnhub 30 calls/sec limit respected with 2-second delays
- **Intelligent Caching**: Prevents redundant API calls and reduces server load
- **Error Recovery**: Failed tasks don't update cache, allowing faster retries
- **File Permissions**: Ensure proper Laravel storage permissions
- **Log Rotation**: Monitor log file sizes and set up rotation
- **Email Notifications**: Consider setting up alerts for critical failures

## 🛠️ Troubleshooting

### Common Issues

#### 1. API Rate Limit Errors
```bash
# Check for rate limit errors
grep "API limit reached" storage/logs/laravel.log

# Solution: The new rate limiting should prevent this
# If still occurring, increase delays in jobs
```

#### 2. Cron Jobs Not Executing
```bash
# Check cron service status
sudo service cron status

# Check system cron logs
sudo tail -f /var/log/cron

# Test manual execution
cd /home/cycloverseworld/domains/themevortex.com/public_html
php artisan cron:queue-manager run-all
```

#### 3. Cache Issues
```bash
# Clear Laravel cache if needed
php artisan cache:clear

# Check cache status in smart scheduler
php artisan cron:queue-manager run-all
```

#### 4. Database Connection Issues
```bash
# Check database connectivity
php artisan tinker
>>> DB::connection()->getPdo();

# Check for database locks
php artisan queue:failed
```

### Performance Optimization

- **Smart Scheduler**: Use `run-all` instead of individual cron jobs
- **Cache Monitoring**: Watch cache hit rates in logs
- **Database Indexing**: Ensure proper indexes on frequently queried tables
- **Log Management**: Rotate logs regularly to prevent disk space issues

### Emergency Commands

```bash
# Force run all tasks (ignores cache)
php artisan cache:clear && php artisan cron:queue-manager run-all

# Check system status
php artisan cron:queue-manager monitor

# Manual stock update (respects rate limits)
php artisan cron:queue-manager dispatch
```
